
############################################################
# PAPER:                                                   #
# An Application of Bayesian Growth Mixture Modelling to   #
# Estimate Infection Incidences from Repeated Serological  #
# Tests                                                    #
# MODEL:                                                   #
# Random Intervention Effects Model                        #
############################################################

#######################################
###  1. Setup Environment Variables ############################################
#######################################

### R version: R.12.1 on Windows XP platform

# Provide path of data files #
ProjectPath = "PATH OF DATA FILES "

# Provide path of temporary file directory for OpenBugs #
setwd(paste(ProjectPath,"Temp\\",sep=""))

### Needed Libraries ###

# Make sure all R libraries are installed

library(R2WinBUGS)
library(BRugs)

#######################################
###  2. Get data & initial values   ############################################
#######################################

### ------------- ###
### 2.1 Read data ###
### ------------- ###

### Subject data ###

# The input dataset "analysisall.csv" contains the following variables:
# - cluster: id of the cluster where the subject lives
# - age: age in months at first survey
# - time2, time2: time in months of survey 2 & 3 (relative to survey 1)
# - INF1, INF2, INF3: infection status at surveys 1, 2, and 3 (for KA cases)
# - elisa1, elisa2, elisa: log-transformed ELISA results
# - Tobsall1, Tobsall2, Tobsall3: time (months) infected before surveys 1 to 3
#     if first time infected, 0 otherwise (for KA cases)
# - INIa1, INIa2, INIa3; INIb1, INIb2, INIb3: two sets of initial values for 
#     INF1, INF2, INF3 (for non-KA cases)
# - TobsallINIa1, TobsallINIa2, TobsallINIa3; TobsallINIb1, TobsallINIb2,
#     TobsallINIb3: two sets of initial values for Tobsall1, Tobsall2, Tobsall3
#     (for non-KA cases)
# - REint1, REslope1; REint2, REslope2: two sets of initial values for the random
#     slopes and intercepts
# - elisa1INIa, elisa2INIa, elisa3INIa; elisa1INIb, elisa2INIb, elisa3INIb: 
#     two sets of initial values for the ELISA data (when missing)

data1 = read.csv(paste(ProjectPath,"analysisall.csv",sep=""))

# Data-subset

# Select random subset to test programs
data1 = data1[seq(1,nrow(data1),20),]

# Variables needed in analysis

sites=data1$cluster
n=nrow(data1)

### Cluster data ###

# The input dataset "clusters.csv" contains the following variables:
# - cluster: id of the cluster
# - intervention: was the cluster in the intervention group (0=No, 1=Yes)
# - pair: randomization pair 

data2 = read.csv(paste(ProjectPath,"clusters.csv",sep=""))

pair = as.numeric(data2$pair)
nsites=length(pair)
intervention=data2$intervention

### --------------------- ###
### 2.2. Needed variables ###
### --------------------- ###

### Select outcome variable ###
Y=cbind(data1$elisa1,data1$elisa2,data1$elisa3)

### Age in months ###
ages=data1$age

### Observation times ###
Tdata=cbind(data1$time1,data1$time2,data1$time3)

### Time of observed KA, if observed ###
Tobs = data1$timeKA 

### Observed infection status ##
Iobs= cbind(data1$INF1,data1$INF2,data1$INF3)

### Observed time infected (prior and during) ###

# - Only needed if for first time infected, set to zero for other times
#   (i.e., if infected the time-point before or later)

Tobsall = cbind(data1$Tobsall1,data1$Tobsall2,data1$Tobsall3)

### Indicator for KA case or not
KA = 1-as.numeric(is.na(data1$timeKA))

### ------------------- ###
### 2.3. Initial values ###
### ------------------- ###

### Initial values for infection status and ###
### time infected relative to current time  ###

IobsINI1=cbind(data1$INIa1,data1$INIa2,data1$INIa3)
IobsINI2=cbind(data1$INIb1,data1$INIb2,data1$INIb3)
TobsallINI1 = cbind(data1$TobsallINIa1,data1$TobsallINIa2,data1$TobsallINIa3)
TobsallINI2 = cbind(data1$TobsallINIb1,data1$TobsallINIb2,data1$TobsallINIb3)

### Initial values for slopes and intercepts ###

int1 = data1$REint1
int2 = data1$REint2
slope1 = data1$REslope1
slope2 = data1$REslope2

### Initial values for ELISA data ###

Yini1=cbind(data1$elisa1INIa,data1$elisa2INIa,data1$elisa3INIa)
Yini2=cbind(data1$elisa1INIb,data1$elisa2INIb,data1$elisa3INIb)

#######################################
###  3. Hidden Markov models        ############################################
#######################################

### ------------- ###
### 3.1 Model     ###
### ------------- ###

WBM <- "BugsModel.txt"

cat("model{                                                                     ",file=WBM)
cat("for (i in 1:Npat){                                                         ",file=WBM, append=T)
cat("  for (j in 1:Ntimes){                                                     ",file=WBM, append=T)
    # Distribution of Elisa
cat("    y[i,j] ~ dt(mu[i,j], tau[i,j],7)                                       ",file=WBM, append=T)
    # Model for Elisa Results
cat("	   mu[i,j]  <- alpha[j] + d[i,j]*beta[i,1] + d[i,j]*delta[i,j]*beta[i,2] +",file=WBM, append=T)
cat("	             ka[i]*d[i,j]*betaKA[1] + ka[i]*d[i,j]*delta[i,j]*betaKA[2]   ",file=WBM, append=T)
cat("	   tau[i,j] <- taufix                                                     ",file=WBM, append=T)
    # Define delta
cat("	   delta[i,j] <- log(max(1,monthsINF[i,j]))                               ",file=WBM, append=T)
    # Disease status 
cat("    d[i,j] ~ dbern(p[i,j])                                                 ",file=WBM, append=T)
cat("    }                                                                      ",file=WBM, append=T)
    # Set random intercept and slope
cat("	 beta[i,1:2] ~ dmnorm(mu.beta[], R[,])I(lim1[1:2],lim2[1:2])              ",file=WBM, append=T)
    # Model for prevalence at study start 
cat("	 logit(p[i,1]) <- gamma0[cluster[i]] + gamma1[cluster[i]]*log(age[i])     ",file=WBM, append=T)
    # Hidden Markov Model 
cat("	 p[i,2] <- d[i,1]+(1-d[i,1])*irate[i,1]                                   ",file=WBM, append=T)
cat("	 p[i,3] <- d[i,2]+(1-d[i,2])*irate[i,2]                                   ",file=WBM, append=T)
    # Hidden Markov Model 
cat("	 logit(irate[i,1]) <- delta1[pair[cluster[i]]]+                           ",file=WBM, append=T)
cat("	                      delta3[pair[cluster[i]]]*intervention[cluster[i]]   ",file=WBM, append=T)
cat("	 logit(irate[i,2]) <- delta1[pair[cluster[i]]]+delta2[pair[cluster[i]]]+  ",file=WBM, append=T)
cat("	                      delta3[pair[cluster[i]]]*intervention[cluster[i]]   ",file=WBM, append=T)
    # Months infected
    # - at first time point: prior infection time
cat("	 monthsINF[i,1] <- Tprior[i,1]*d[i,1]                                     ",file=WBM, append=T)
    # - at second time point: for those prior infected: previous time + period between 2 visits
    #                         for those newly infected: Tprior 1 or 2
cat("	 monthsINF[i,2] <- monthsINF[i,1] + period[i,1]*d[i,1] +                  ",file=WBM, append=T)
cat("	               (1-d[i,1])*d[i,2] * Tprior[i,2]                            ",file=WBM, append=T)
cat("	 monthsINF[i,3] <- monthsINF[i,2] + 12*d[i,2] +                           ",file=WBM, append=T)
cat("	               (1-d[i,2])*d[i,3] * Tprior[i,3]                            ",file=WBM, append=T)
##### ----- PRIORS ----- #####
    # Priors for previous infection time
cat("	 Tprior[i,1] ~ dunif(0,age[i])                                            ",file=WBM, append=T)
cat("	 Tprior[i,2] ~ dunif(0,period[i,1])                                       ",file=WBM, append=T)
cat("	 Tprior[i,3] ~ dunif(0,period[i,2])                                       ",file=WBM, append=T)
cat(" }                                                                         ",file=WBM, append=T)
    # Priors for disease probabilities and infection rate
cat("for (k in 1:Nc){                                                           ",file=WBM, append=T)
cat("    gamma0[k] ~ dnorm(mu.gamma0,tau.gamma0)                                ",file=WBM, append=T)
cat("    gamma1[k] ~ dnorm(mu.gamma1,tau.gamma1)                                ",file=WBM, append=T)
cat(" }                                                                         ",file=WBM, append=T)
cat(" mu.gamma0 ~ dt(-4,.01,7)                                                  ",file=WBM, append=T)
cat(" mu.gamma1 ~ dt(0,.16,7)                                                   ",file=WBM, append=T)
cat(" tau.gamma0 <- pow(sigma.gamma0,-2)                                        ",file=WBM, append=T)
cat(" sigma.gamma0 ~ dnorm(0,.01)I(0,)                                          ",file=WBM, append=T)
cat(" tau.gamma1 <- pow(sigma.gamma1,-2)                                        ",file=WBM, append=T)
cat(" sigma.gamma1 ~ dnorm(0,.01)I(0,)                                          ",file=WBM, append=T)
cat(" for (k in 1:Np){                                                          ",file=WBM, append=T)
cat("    delta1[k] ~ dnorm(mu.delta1,tau.delta1)                                ",file=WBM, append=T)
cat("    delta2[k] ~ dnorm(mu.delta2,tau.delta2)                                ",file=WBM, append=T)
cat("    delta3[k] ~ dnorm(mu.delta3,tau.delta3)                                ",file=WBM, append=T)
cat(" }                                                                         ",file=WBM, append=T)
cat(" mu.delta1 ~ dt(-4,0.01,7)                                                 ",file=WBM, append=T)
cat(" mu.delta2 ~ dt(0,0.16,7)                                                  ",file=WBM, append=T)
cat(" mu.delta3 ~ dt(0,0.16,7)                                                  ",file=WBM, append=T)
cat(" tau.delta1 <- pow(sigma.delta1,-2)                                        ",file=WBM, append=T)
cat(" sigma.delta1 ~ dnorm(0,.01)I(0,)                                          ",file=WBM, append=T)
cat(" tau.delta2 <- pow(sigma.delta2,-2)                                        ",file=WBM, append=T)
cat(" sigma.delta2 ~ dnorm(0,.01)I(0,)                                          ",file=WBM, append=T)
cat(" tau.delta3 <- pow(sigma.delta3,-2)                                        ",file=WBM, append=T)
cat(" sigma.delta3 ~ dnorm(0,.01)I(0,)                                          ",file=WBM, append=T)
    # Priors for parameters of Y
    # - intercepts prior to infection
cat("	alpha[1] ~ dnorm( palpha, 4)                                              ",file=WBM, append=T)
cat("	alpha[2] <- alpha[1]+dif[1]                                               ",file=WBM, append=T)
cat("	alpha[3] <- alpha[1]+dif[2]                                               ",file=WBM, append=T)
cat("	dif[1] ~ dnorm( 0, 10)                                                    ",file=WBM, append=T)
cat("	dif[2] ~ dnorm( 0, 10)                                                    ",file=WBM, append=T)
    # - after infection
cat(" mu.beta[1:2] ~ dmnorm(pbeta[],prec[,])                                    ",file=WBM, append=T)
cat(" R[1:2,1:2] ~ dwish(Omega[,], DF)	                                        ",file=WBM, append=T)
    # - measurement error
cat("	taufix <- pow(sigma,-2)                                                   ",file=WBM, append=T)
cat("	sigma ~ dunif(0,.8)                                                       ",file=WBM, append=T)
    # - after KA
cat(" betaKA[1:2] ~ dmnorm(pbetaKA[],precKA[,])                                 ",file=WBM, append=T)
cat("}                                                                          ",file=WBM, append=T)

### ---------------------------------- ###
### 3.2 Data, Initial values en Output ###
### ---------------------------------- ###

# Setup data (including priors given as data)
DPo = list(Npat=n,
           Ntimes=3,
           Nc=nsites,
           Np=nsites/2,
           y=Y,
           d=Iobs,
           ka=KA,
           cluster=sites,
           pair=pair,
           intervention=intervention,
           age = ages,
           Tprior=Tobsall,
           period=cbind(data1$time2,data1$time3-data1$time2),
           # Priors provided as data:
           # - alpha (intercept prior to infection)
           palpha=-2,
           # - beta (intercept and slope after infection)
           pbeta=c(1.5,-.2),
           prec=matrix(c(4,0,0,100),nrow=2,byrow=T),
           # (constraints on beta)
           lim1=c(0,-5),
           lim2=c(5,0),
           # - random variation of intercept and slope after infection
           DF=2,
           Omega=matrix(c(.01,0,0,.01),nrow=2,byrow=T),
           # - difference asymptomatics/symptomatics
           pbetaKA=c(.25,-.05),
           precKA=matrix(c(20,0,0,10),nrow=2,byrow=T)
           )
# Initial values
I1Po = list(y=Yini1,
            d=IobsINI1,
            Tprior=TobsallINI1,
            alpha=c(-2.5,NA,NA),
            dif=c(.1,.1),
            mu.beta=c(2,-.15),
            sigma=.2,
            R=matrix(c(5,0,0,100),nrow=2,byrow=T),
            beta=cbind(int1,slope1),
            gamma0=rep(-7,nsites),
            gamma1=rep(1.2,nsites),
            delta1=rep(-3,nsites/2),
            delta2=rep(-.5,nsites/2),
            delta3=rep(-1,nsites/2),
            sigma.delta1=.5,sigma.delta2=.5,sigma.delta3=.5,
            mu.delta1=-3,mu.delta2=-.5,mu.delta3=-.5,
            sigma.gamma0=.4,sigma.gamma1=.05,
            mu.gamma0=-7,mu.gamma1=1,
            betaKA=c(0.3,-.05)
            )
I2Po = list(y=Yini2,
            d=IobsINI2,
            Tprior=TobsallINI2,
            alpha=c(-1.5,NA,NA),
            dif=c(-.1,-.1),
            mu.beta=c(1,-.2),
            sigma=.3,
            R=matrix(c(10,0,0,500),nrow=2,byrow=T),
            beta=cbind(int2,slope2),
            gamma0=rep(-8,nsites),
            gamma1=rep(.8,nsites),
            delta1=rep(-4,nsites/2),
            delta2=rep(.5,nsites/2),
            delta3=rep(1,nsites/2),
            sigma.delta1=1,sigma.delta2=1,sigma.delta3=1,
            mu.delta1=-4,mu.delta2=.5,mu.delta3=.5,
            sigma.gamma0=.2,sigma.gamma1=.3,
            mu.gamma0=-8,mu.gamma1=.5,
            betaKA=c(0.1,-.01)
            )
IPo = list(I1Po,I2Po)

# Setup parameters to save
PTS <- c("alpha","mu.beta","R",
         "sigma",
         "mu.delta1","mu.delta2","mu.delta3","delta3",
         "sigma.delta1","sigma.delta2","sigma.delta3",
         "mu.gamma0","mu.gamma1",
         "sigma.gamma0","sigma.gamma1","betaKA")

### ------------- ###
### 3.3 Run model ###
### ------------- ###

modelCheck(WBM)
modelData(bugsData(DPo))
modelCompile(numChains=2)
bugsInits(IPo,c(paste(ProjectPath,"Temp\\inits1.txt",sep=""),paste(ProjectPath,"Temp\\inits2.txt",sep="")),numChains=2)
modelInits(paste(ProjectPath,"Temp\\inits1.txt",sep=""),1)
modelInits(paste(ProjectPath,"Temp\\inits2.txt",sep=""),2)
modelGenInits()

modelUpdate(1000,overRelax=T)
modelUpdate(1000)

samplesSetThin(1)
samplesSet(PTS)

modelUpdate(1000)                               

model.sims = samplesStats("*")
model.mcmc = buildMCMC("*")

